#!/bin/bash
set -e

TEMPLATE_URL="https://cloud-images.ubuntu.com/plucky/current/plucky-server-cloudimg-amd64.img"
VMID=9000
TEMPLATE_NAME="ubuntu-25.04-cloudimg-temp"
STORAGE="local-lvm"
SKIP_DISK_SETUP=false

# Parser avec getopt
OPTS=$(getopt -o "" --long vmid:,template:,storage:,skip-disk -n 'script' -- "$@")
eval set -- "$OPTS"

while true; do
    case "$1" in
        --vmid) VMID="$2"; shift 2 ;;
        --template) TEMPLATE_NAME="$2"; shift 2 ;;
        --storage) STORAGE="$2"; shift 2 ;;
        --skip-disk) SKIP_DISK_SETUP=true; shift ;;
        --) shift; break ;;
        *) break ;;
    esac
done

# Récupérer les clés SSH
SSH_KEYS=$(curl -s https://gist.vorpax.dev/vorpax/ssh/raw/HEAD/ssh-keys.pub | sed 's/%20/ /g; s/%0A/\n/g; s/%3D/=/g; s/%2F/\//g; s/%40/@/g; s/%3A/:/g')

cd /tmp

if [ "$SKIP_DISK_SETUP" = false ]; then
    echo "Downloading cloud image..."
    wget -O "${TEMPLATE_NAME}_disk" "$TEMPLATE_URL"
    
    # Créer la VM avec une config minimale
    qm create "$VMID" \
      --name "$TEMPLATE_NAME" \
      --memory 3072 \
      --cores 3 \
      --scsihw virtio-scsi-pci \
      --serial0 socket \
      --vga serial0 \
      --net0 virtio,bridge=vmbr0
    
    # Importer le disque
    echo "Importing disk..."
    qm importdisk "$VMID" "${TEMPLATE_NAME}_disk" "$STORAGE"
    
    # Extraire le chemin complet du disque unused (la partie APRÈS "unused0: ")
    DISK_PATH=$(qm config "$VMID" | grep "^unused0:" | awk '{print $2}')
    
    if [ -z "$DISK_PATH" ]; then
        echo "ERROR: Could not find unused disk"
        exit 1
    fi
    
    echo "Attaching disk: $DISK_PATH"
    qm set "$VMID" --scsi0 "$DISK_PATH"
    
    rm "${DISK_PATH}"
else
    echo "Skipping disk setup (--skip-disk flag)"
fi

# Créer le dossier des clés SSH
SSH_KEYS_DIR="ssh_keys_${VMID}"
mkdir -p "$SSH_KEYS_DIR"

# Écrire toutes les clés dans un seul fichier
SSH_KEY_FILE="${SSH_KEYS_DIR}/authorized_keys"
echo "$SSH_KEYS" > "$SSH_KEY_FILE"
echo "SSH keys written to $SSH_KEY_FILE"

# Configurer cloud-init et le reste
qm set "$VMID" \
  --sshkey "$SSH_KEY_FILE" \
  --ide2 "${STORAGE}:cloudinit" \
  --boot order=scsi0 \
  --ciuser packer \
  --searchdomain .lab.internal \
  --ipconfig0 ip=dhcp

# Convertir en template
qm template "$VMID"

echo "Template $VMID ready for Packer"