#!/bin/bash
set -e

TEMPLATE_URL="https://cloud-images.ubuntu.com/plucky/current/plucky-server-cloudimg-amd64.img"
VMID=9001
TEMPLATE_NAME="ubuntu-25.04-cloudimg-temp"
STORAGE="local-lvm"
SKIP_DISK_SETUP=false
VMID_CUSTOM=false  # Track if VMID was explicitly set

# Parser avec getopt
OPTS=$(getopt -o "" --long vmid:,template:,storage:,skip-disk -n 'script' -- "$@")
eval set -- "$OPTS"

while true; do
    case "$1" in
        --vmid) VMID="$2"; VMID_CUSTOM=true; shift 2 ;;
        --template) TEMPLATE_NAME="$2"; shift 2 ;;
        --storage) STORAGE="$2"; shift 2 ;;
        --skip-disk) SKIP_DISK_SETUP=true; shift ;;
        --) shift; break ;;
        *) break ;;
    esac
done

# Detect if we need sudo (non-root user)
if [ "$(id -u)" -ne 0 ]; then
    echo "Running as non-root user, will use sudo for qm commands"
    SUDO="sudo"
else
    SUDO=""
fi

# Check if VMID already exists
ORIGINAL_VMID=$VMID
echo "Checking if VMID $VMID is available..."
if $SUDO qm status "$VMID" &>/dev/null; then
    # VMID exists, check if it's already a template
    IS_TEMPLATE=$($SUDO qm config "$VMID" | grep -q "^template:" && echo "yes" || echo "no")

    if [ "$IS_TEMPLATE" = "yes" ]; then
        echo "✓ VMID $VMID already exists and is a template (idempotent)"
        echo ""
        echo "========================================="
        echo "Template already exists!"
        echo "  VMID: $VMID"
        echo "========================================="
        echo ""
        echo "Skipping template creation (already exists)."
        exit 0
    else
        echo "⚠ VMID $VMID already exists but is NOT a template!"

        # If VMID is the default (9001) and was not explicitly set, error out
        # This ensures Packer can find the expected template
        if [ "$VMID_CUSTOM" = "false" ]; then
            echo ""
            echo "========================================="
            echo "ERROR: Default VMID $VMID is occupied!"
            echo "========================================="
            echo ""
            echo "VMID $VMID exists but is not a template."
            echo ""
            echo "Solutions:"
            echo "  1. Delete the VM: qm destroy $VMID"
            echo "  2. Convert it to template: qm template $VMID"
            echo "  3. Use a different VMID: $0 --vmid XXXX"
            echo ""
            exit 1
        fi

        # If VMID was explicitly set, auto-increment to next available
        echo "Finding next available VMID..."
        MAX_VMID=$($SUDO qm list | awk 'NR>1 {print $1}' | sort -n | tail -1)

        if [ -z "$MAX_VMID" ]; then
            # No VMs exist, use the requested VMID
            echo "No existing VMs found, using VMID $VMID"
        else
            # Use max VMID + 1
            VMID=$((MAX_VMID + 1))
            echo "✓ Using next available VMID: $VMID (max existing: $MAX_VMID)"
        fi
    fi
else
    echo "✓ VMID $VMID is available"
fi

# Récupérer les clés SSH
SSH_KEYS=$(curl -s https://gist.vorpax.dev/vorpax/ssh/raw/HEAD/ssh-keys.pub | sed 's/%20/ /g; s/%0A/\n/g; s/%3D/=/g; s/%2F/\//g; s/%40/@/g; s/%3A/:/g')

cd /tmp

if [ "$SKIP_DISK_SETUP" = false ]; then
    echo "Downloading cloud image..."
    wget -O "${TEMPLATE_NAME}_disk" "$TEMPLATE_URL"
    
    # Créer la VM avec une config minimale
    $SUDO qm create "$VMID" \
      --name "$TEMPLATE_NAME" \
      --memory 3072 \
      --cores 3 \
      --scsihw virtio-scsi-pci \
      --serial0 socket \
      --vga serial0 \
      --net0 virtio,bridge=vmbr0

    # Importer le disque
    echo "Importing disk..."
    $SUDO qm importdisk "$VMID" "${TEMPLATE_NAME}_disk" "$STORAGE"

    # Extraire le chemin complet du disque unused (la partie APRÈS "unused0: ")
    DISK_PATH=$($SUDO qm config "$VMID" | grep "^unused0:" | awk '{print $2}')

    if [ -z "$DISK_PATH" ]; then
        echo "ERROR: Could not find unused disk"
        exit 1
    fi

    echo "Attaching disk: $DISK_PATH"
    $SUDO qm set "$VMID" --scsi0 "$DISK_PATH"

    # Cleanup temporary downloaded file
    rm -f "${TEMPLATE_NAME}_disk"
else
    echo "Skipping disk setup (--skip-disk flag)"
fi

# Créer le dossier des clés SSH
SSH_KEYS_DIR="ssh_keys_${VMID}"
mkdir -p "$SSH_KEYS_DIR"

# Écrire toutes les clés dans un seul fichier
SSH_KEY_FILE="${SSH_KEYS_DIR}/authorized_keys"
echo "$SSH_KEYS" > "$SSH_KEY_FILE"
echo "SSH keys written to $SSH_KEY_FILE"

# Configurer cloud-init et le reste
$SUDO qm set "$VMID" \
  --sshkey "$SSH_KEY_FILE" \
  --ide2 "${STORAGE}:cloudinit" \
  --boot order=scsi0 \  
  --ciuser packer \
  --searchdomain .lab.internal \
  --ipconfig0 ip=dhcp
  --bios ovmf

# Convertir en template
$SUDO qm template "$VMID"

echo ""
echo "========================================="
echo "✓ Template created successfully!"
echo "  VMID: $VMID"
echo "  Name: $TEMPLATE_NAME"
echo "========================================="
echo ""
echo "Template is ready for Packer to clone and provision."